<?php

/**
 * @package probance-optin 
 */

/**
 * Plugin Name: Probance-optin
 * Plugin URI: https://www.probance.com/
 * Description: Plugin displaying a checkbox, a Newsletter banner and a popup to manage the optin on Probance side 
 * Version: 1.8.6
 * Author: Probance
 * Author URI: https://www.probance.com/
 **/

defined('ABSPATH') or die('Access denied.');

if (file_exists(dirname(__FILE__) . '/vendor/autoload.php')) {
	require_once dirname(__FILE__) . '/vendor/autoload.php';
}

use ProbanceOptin\Init;

define('PLUGIN_OPTIN_PATH', plugin_dir_path(__FILE__));
define('PLUGIN_OPTIN_URL', plugin_dir_url(__FILE__));
define('NEWSLETTER_ATTR', array('name' => 'probance_newsletter', 'shortcode' => 'probance_newsletter'));
define('PROB_DEBUG', get_option('probance-optin_api-cbdebug'));
if (!defined('PLUGIN_OPTIN_UPDATE_CACHE_TTL')) {
    define('PLUGIN_OPTIN_UPDATE_CACHE_TTL', DAY_IN_SECONDS);
}

add_action('admin_init', function() {
    add_filter('site_transient_update_plugins', 'plugin_optin_check_for_update');
    add_filter('plugins_api', 'plugin_optin_plugins_api_handler', 10, 3);
});


function plugin_optin_check_for_update($transient) {
    // Sécurité de base
    if (!is_object($transient)) {
        $transient = new stdClass();
    }

    $plugin_file = 'probance-optin/probance-optin.php';
    $cache_key   = 'plugin_optin_remote_data';

    // ------------------------------------------------------------------
    // 🔧 Mode test : on veut vérifier souvent -> cache = 5 minutes
    // ------------------------------------------------------------------
    $remote = get_transient($cache_key);

    if ($remote === false) {
        // Log pour vérifier les appels distants (visible dans wp-content/debug.log)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Probance Optin] Appel distant à ' . date('Y-m-d H:i:s'));
        }

        // Appel distant
        $response = wp_remote_get(
            "https://download.woocommerce-app.probance.com/probance-optin/config.json",
            [
                'timeout' => 5,
                'headers' => ['Accept' => 'application/json']
            ]
        );

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $remote = json_decode(wp_remote_retrieve_body($response));
            
            // Mise en cache
            set_transient($cache_key, $remote, PLUGIN_OPTIN_UPDATE_CACHE_TTL);
        } else {
            // Erreur de connexion ou de réponse : on ne casse pas le site
            return $transient;
        }
    }

    // ------------------------------------------------------------------
    // 🧠 Validation du JSON distant
    // ------------------------------------------------------------------
    if (!$remote || empty($remote->version)) {
        return $transient;
    }

    // Charger la liste des plugins installés
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
    $plugins = get_plugins();

    if (empty($plugins[$plugin_file])) {
        return $transient;
    }

    $current_version = $plugins[$plugin_file]['Version'];

    // ------------------------------------------------------------------
    // 🔍 Comparaison des versions
    // ------------------------------------------------------------------
    if (version_compare($current_version, $remote->version, '<')) {
        $transient->response[$plugin_file] = (object)[
            'slug'        => $remote->slug ?? 'probance-optin',
            'plugin'      => $plugin_file,
            'new_version' => $remote->version,
            'tested'      => get_bloginfo('version'),
            'package'     => $remote->download_url ?? '',
            'url'         => $remote->homepage ?? '',
        ];

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Probance Optin] Nouvelle version dispo : ' . $remote->version);
        }
    } else {
        // Pas de nouvelle version
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Probance Optin] Version à jour : ' . $current_version);
        }
    }

    return $transient;
}


function plugin_optin_plugins_api_handler($result, $action, $args) {
    if ($action !== 'plugin_information' || $args->slug !== 'probance-optin') return $result;

    $remote = wp_remote_get('https://download.woocommerce-app.probance.com/probance-optin/config.json');

    if (!is_wp_error($remote) && $remote['response']['code'] === 200) {
        $remote_data = json_decode($remote['body']);

        return (object)[
            'name' => $remote_data->name,
            'slug' => 'probance-optin',
            'version' => $remote_data->version,
            'author' => $remote_data->author,
            'homepage' => $remote_data->homepage,
            'sections' => [
                'description' => $remote_data->description,
                'changelog' => $remote_data->changelog,
            ],
            'download_link' => $remote_data->download_url,
        ];
        
    }

    return $result;
}

add_action('plugins_loaded', function() {
    try {
        if (class_exists('ProbanceOptin\\Init')) {
            Init::register_services();
        } else {
            throw new Exception('Init class is missing.');
        }
    } catch (Throwable $e) {

        // Message visible uniquement en admin
        add_action('admin_notices', function() use ($e) {
            echo '<div class="notice notice-error"><p>';
            echo 'Le plugin <strong>Probance Optin</strong> a rencontré une erreur et n’a pas pu s’initialiser : ';
            echo esc_html($e->getMessage());
            echo '</p></div>';
        });

        // Sécurité : désactive automatiquement le plugin pour éviter de bloquer le site
        deactivate_plugins(plugin_basename(__FILE__));
    }
});
