<?php

/**
 * @package probance-optin
 */

namespace ProbanceOptin\Common;

final class Utils
{

    /**
     * @return boolean : true if a translation exists for the current locale user  
     */
    public static function translationExists($lang)
    {
        return in_array($lang, self::getLanguages());
    }

    /**
     * Return the current user locale
     * @return string $user_locale : string format 2-char lowercase. Empty if no language is detected.
     */
    public static function getUserLanguage()
    {
        $user_locale = "";

        $user = wp_get_current_user();

        $user_locale_unformated = get_user_meta($user->ID, 'locale', true);

        self::write_log("Language unformated for user ($user->ID) : $user_locale_unformated.");

        if (preg_match('/^[a-z]{2}(?:_[A-Z]{2})?$/', $user_locale_unformated, $matches) === false) {
            self::write_log("No language detected for the current user ($user->ID).");
        } else {
            switch (true) {
                case (preg_match('/^[a-z]{2}_[A-Z]{2}$/', $user_locale_unformated, $matches) == 1):
                    $user_locale = explode('_', $user_locale_unformated)[0];
                    break;

                case (preg_match('/^[a-z]{2}$/', $user_locale_unformated, $matches) == 1):
                    $user_locale = $user_locale_unformated;
                    break;
            }
        }

        self::write_log("Language for user ($user->ID) : $user_locale.");

        return $user_locale;
    }

    public static function get_user_preferred_language()
    {
        $accepted_languages = $_SERVER['HTTP_ACCEPT_LANGUAGE'];
        $preferred_language = '';

        // Parse the accepted languages header
        $languages = explode(',', $accepted_languages);

        if (!empty($languages)) {
            // Extract the first language from the list
            $user_locale_unformated = strtolower(strtok($languages[0], ';'));

            if (preg_match('/^[a-zA-Z]{2}(?:[-_]([a-zA-Z]{2}))?$/', $user_locale_unformated, $matches) === false) {
                self::write_log("No language detected for the current user ($user->ID).");
            } else {
                switch (true) {
                    case (preg_match('/^[a-zA-Z]{2}(?:[-_]([a-zA-Z]{2}))?$/', $user_locale_unformated, $matches) == 1):
                        $preferred_language = explode('-', $user_locale_unformated)[0];
                        break;

                    case (preg_match('/^[a-z]{2}$/', $user_locale_unformated, $matches) == 1):
                        $preferred_language = $user_locale_unformated;
                        break;
                }
            }
        }

        return $preferred_language;
    }

    public static function get_user_locale_by_ip()
    {
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $locale = '';

        // Use a service to determine user's country based on IP
        $response = wp_remote_get("http://ip-api.com/json/$ip_address");

        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response));

            if ($data && isset($data->countryCode)) {
                $locale = strtolower($data->countryCode);
            }
        }

        return $locale;
    }

    /**
     * Return languages available for current website 
     * @return array $langs : array of string format 2-char to uppercase. Empty array if no language detected.
     */
    public static function getLanguages()
    {

        $langs = [];

        // Admin Language
        $admin_lang = get_bloginfo('language');

        // WPML languages available for this site

        // $wpml_langs = array( "es" => array(
        //         "id" => 25,
        //         "active" => 1,
        //         "default_locale" => "es_ES",
        //         "native_name" => "Español",
        //         "missing" => 0,
        //         "translated_name" => "Espagnol",
        //         "language_code" => "es",
        //         "country_flag_url" => "http://yourdomain/wpmlpath/res/flags/it.png",
        //         "url" => "http://yourdomain/it/circa"
        //         ),
        //         "en" => array(
        //             "id" => 26,
        //             "active" => 0,
        //             "default_locale" => "en_EN",
        //             "native_name" => "English",
        //             "missing" => 0,
        //             "translated_name" => "Anglais",
        //             "language_code" => "en",
        //             "country_flag_url" => "http://yourdomain/wpmlpath/res/flags/it.png",
        //             "url" => "http://yourdomain/it/circa"
        //         ),
        //         "it" => array(
        //             "id" => 27,
        //             "active" => 0,
        //             "default_locale" => "it_IT",
        //             "native_name" => "Italiano",
        //             "missing" => 0,
        //             "translated_name" => "Italian",
        //             "language_code" => "it",
        //             "country_flag_url" => "http://yourdomain/wpmlpath/res/flags/it.png",
        //             "url" => "http://yourdomain/it/circa"
        //         )
        //         );

        $wpml_langs = apply_filters('wpml_active_languages', NULL);

        if ($admin_lang) {
            $admin_lang = strtoupper(explode('-', $admin_lang)[0]);

            array_push($langs, $admin_lang);
        }

        if ($wpml_langs) {
            $wpml_langs_keys = [];

            foreach (array_keys($wpml_langs) as $l) {
                array_push($wpml_langs_keys, strtoupper($l));
            }

            $langs = array_merge($langs, $wpml_langs_keys);
        }

        if (self::option_exists('probance-optin-languages')) {
            $opt_langs_str = get_option('probance-optin-languages');

            $opt_langs_arr = explode(';', $opt_langs_str);

            $langs = array_merge($langs, $opt_langs_arr);
        }

        return array_unique($langs);
    }

    public static function write_log($log)
    {

        // No more log in server side since the version 1.8.4
        // if (PROB_DEBUG == 1) {
        //     if (is_array($log) || is_object($log)) {
        //         error_log(print_r($log, true));
        //     } else {
        //         error_log($log);
        //     }
        // }

        // For future use, we can log in woocommerce logger if available in the context and not log in the error log everytime
        // if (function_exists('wc_get_logger')) {
        //     $logger  = wc_get_logger();
        //     $context = array('source Log' => 'Probance Optin');
        //     $logger->debug($log, $context);
        // } else {

        //     if (is_array($log) || is_object($log)) {
        //         error_log(print_r($log, true));
        //     } else {
        //         error_log($log);
        //     }
        // }
    }

    /**
     * Check if an option exists 
     * @param $arg string id of the option
     * @return true|false boolean 
     */
    public static function option_exists($arg)
    {

        global $wpdb;
        $prefix = $wpdb->prefix;
        $db_options = $prefix . 'options';
        $sql_query = 'SELECT * FROM ' . $db_options . ' WHERE option_name LIKE "' . $arg . '"';

        $results = $wpdb->get_results($sql_query, OBJECT);

        if (count($results) === 0) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Test api contact with provided parameter or 
     * with default parameter from woocommerce saved options
     * Test that the email exists in rt db
     * @return array result from the api contact
     */
    public static function test_api_contact_working($project_name = null, $api_account = null, $api_login = null, $api_password = null, $api_server = null, $email = null)
    {

        // Get default values from woocommerce options
        $default_values = array(
            'project_name' => get_option('probance-optin_api-projectName'),
            'api_account' => get_option('probance-optin_api-account'),
            'api_login' => get_option('probance-optin_api-login'),
            'api_password' => get_option('probance-optin_api-passwd'),
            'api_server' => get_option('probance-optin_api-infra'),
            'email' => 'probance.test@probance.com'
        );


        // Create an array with arguments function. argName => argValue
        $parameters = compact('project_name', 'api_account', 'api_login', 'api_password', 'api_server', 'email');

        Utils::write_log('[PROBANCE - optin - test api contact] default value: ' . implode('|', $default_values));
        Utils::write_log('[PROBANCE - optin - test api contact] parameters: ' . implode('|', $parameters));

        // Replace null values with default values
        foreach ($parameters as $param => $value) {
            if ($value == null) {
                $parameters[$param] = $default_values[$param];
            }
        }

        Utils::write_log('[PROBANCE - optin - test api contact] parameters after settings default: ' . implode('|', $parameters));

        $api_contact_url = 'https://' . $parameters['project_name'] . '.' . $parameters['api_server'] . '/rt/api/resource/client/' . $parameters['project_name'] . '_' . $parameters['api_account'] . '/search?email=' . $parameters['email'];

        Utils::write_log('[PROBANCE - optin - test api contact] api contact url: ' . $api_contact_url);

        //Mise en place de la ressource curl
        $options = array(
            CURLOPT_USERPWD => $parameters['api_login'] . ':' . $parameters['api_password'],
            CURLOPT_URL => $api_contact_url,
            CURLOPT_HTTPHEADER => array('accept:application/json'),
            CURLOPT_RETURNTRANSFER => 1, //permet de récupérer la réponse dans une variable
            CURLOPT_HEADER => 0, //on ne retourne pas les header
            CURLOPT_NOBODY => 0, //on ne retourne pas le body
            CURLOPT_CONNECTTIMEOUT => 120,
            CURLOPT_TIMEOUT => 120,
        );

        $ch = curl_init();
        curl_setopt_array($ch, $options);

        //Exécution de l'appel
        if (!$result = curl_exec($ch)) {
            trigger_error(curl_error($ch));
        }

        //On récupère le code HTTP de la réponse
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        Utils::write_log('[PROBANCE - optin - test api contact] http return code: ' . $httpcode);

        // close cURL resource, and free up system resources
        curl_close($ch);
        Utils::write_log('[PROBANCE - optin - test api contact] result: ' . $result);

        return array('http_code' => $httpcode, 'api_contact' => json_decode($result) ? json_decode($result) : $result);
    }
}
