<?php

/**
 * @package probance-optin 
 */

/**
 * Plugin Name: Probance-optin
 * Plugin URI: https://www.probance.com/
 * Description: Plugin displaying a checkbox, a Newsletter banner and a popup to manage the optin on Probance side 
 * Version: 1.8.4
 * Author: Probance
 * Author URI: https://www.probance.com/
 **/

defined('ABSPATH') or die('Access denied.');

if (file_exists(dirname(__FILE__) . '/vendor/autoload.php')) {
	require_once dirname(__FILE__) . '/vendor/autoload.php';
}

use ProbanceOptin\Init;

define('PLUGIN_OPTIN_PATH', plugin_dir_path(__FILE__));
define('PLUGIN_OPTIN_URL', plugin_dir_url(__FILE__));
define('NEWSLETTER_ATTR', array('name' => 'probance_newsletter', 'shortcode' => 'probance_newsletter'));
define('PROB_DEBUG', get_option('probance-optin_api-cbdebug'));

add_action('admin_init', function() {
    add_filter('site_transient_update_plugins', 'plugin_optin_check_for_update');
    add_filter('plugins_api', 'plugin_optin_plugins_api_handler', 10, 3);
});

function plugin_optin_check_for_update($transient) {

    if (!is_object($transient)) {
        $transient = new stdClass();
    }
    
    if (empty($transient->checked)) {
        $plugins = get_plugins();
        foreach ($plugins as $file => $data) {
            if (strpos($file, 'probance-optin.php') !== false) {
                $transient->checked[$file] = $data['Version']; 
                break;
            }
        }
    }

    // Vérifie que le transient est bien structuré
    if (empty($transient->checked)) {
        #error_log('[Probance Optin] Transient->checked est vide');
        return $transient;
    }

    // Charger la liste des plugins
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
    $plugins = get_plugins();

    // Log des plugins trouvés
    #error_log('[Probance Optin] Plugins détectés: ' . print_r(array_keys($plugins), true));

    $plugin_file = null;
    foreach ($plugins as $file => $data) {
        if (strpos($file, 'probance-optin') !== false) {
            $plugin_file = $file; // ex: probance-optin/probance-optin.php
            break;
        }
    }

    if (!$plugin_file) {
        #error_log('[Probance Optin] Plugin introuvable dans la liste des plugins');
        return $transient;
    }

    #error_log('[Probance Optin] Plugin détecté: ' . $plugin_file);

    // Appel au JSON distant
    $remote = wp_remote_get("https://download.woocommerce-app.probance.com/probance-optin/config.json", [
        'timeout' => 10,
        'headers' => ['Accept' => 'application/json']
    ]);

    if (is_wp_error($remote)) {
        #error_log('[Probance Optin] Erreur WP lors de la récupération du JSON distant: ' . $remote->get_error_message());
        return $transient;
    }

    $http_code = wp_remote_retrieve_response_code($remote);
    #error_log('[Probance Optin] Code HTTP reçu: ' . $http_code);

    if ($http_code !== 200) {
        #error_log('[Probance Optin] Le JSON distant n\'a pas répondu correctement.');
        return $transient;
    }

    $remote_body = wp_remote_retrieve_body($remote);
    #error_log('[Probance Optin] Contenu brut du JSON: ' . $remote_body);

    $remote = json_decode($remote_body);
    if (!$remote || empty($remote->version)) {
        #error_log('[Probance Optin] Impossible de parser le JSON ou version absente.');
        return $transient;
    }

    // Versions
    $plugin_data     = $plugins[$plugin_file];
    $current_version = $plugin_data['Version'];
    $remote_version  = $remote->version;

    #error_log("[Probance Optin] Version actuelle: $current_version");
    #error_log("[Probance Optin] Version distante: $remote_version");

    // Comparaison
    if (version_compare($current_version, $remote_version, '<')) {
        $transient->response[$plugin_file] = (object)[
            'slug'        => $remote->slug,
            'plugin'      => $plugin_file,
            'new_version' => $remote->version,
            'tested'      => get_bloginfo('version'),
            'package'     => $remote->download_url,
            'url'         => $remote->homepage,
        ];

        #error_log('[Probance Optin] Nouvelle version détectée: ' . $remote_version);
    }

    #error_log('[Probance Optin] Transient final: ' . print_r($transient, true));

    return $transient;
}


function plugin_optin_plugins_api_handler($result, $action, $args) {
    if ($action !== 'plugin_information' || $args->slug !== 'probance-optin') return $result;

    $remote = wp_remote_get('https://download.woocommerce-app.probance.com/probance-optin/config.json');

    if (!is_wp_error($remote) && $remote['response']['code'] === 200) {
        $remote_data = json_decode($remote['body']);

        return (object)[
            'name' => $remote_data->name,
            'slug' => 'probance-optin',
            'version' => $remote_data->version,
            'author' => $remote_data->author,
            'homepage' => $remote_data->homepage,
            'sections' => [
                'description' => $remote_data->description,
                'changelog' => $remote_data->changelog,
            ],
            'download_link' => $remote_data->download_url,
        ];
        
    }

    return $result;
}

add_action('plugins_loaded', function() {
    try {
        if (class_exists('ProbanceOptin\\Init')) {
            Init::register_services();
        } else {
            throw new Exception('Init class is missing.');
        }
    } catch (Throwable $e) {

        // Message visible uniquement en admin
        add_action('admin_notices', function() use ($e) {
            echo '<div class="notice notice-error"><p>';
            echo 'Le plugin <strong>Probance Optin</strong> a rencontré une erreur et n’a pas pu s’initialiser : ';
            echo esc_html($e->getMessage());
            echo '</p></div>';
        });

        // Sécurité : désactive automatiquement le plugin pour éviter de bloquer le site
        deactivate_plugins(plugin_basename(__FILE__));
    }
});
